<?php
session_start();

// 🔒 Cegah akses tanpa login
if (!isset($_SESSION['id_admin'])) {
    header("Location: ../login_admin.php");
    exit();
}

require_once("../includes/db_connect.php");

// Ambil data rekam medis dengan join ke tabel pasien, layanan, dan pegawai
$query = "SELECT 
            rm.id_rekam,
            rm.keluhan,
            rm.tindakan,
            rm.hasil,
            rm.tanggal_periksa,
            p.nama_pasien,
            l.nama_layanan,
            pg.nama_pegawai
          FROM rekam_medis rm
          LEFT JOIN data_diri_pasien p ON rm.id_pasien = p.id_pasien
          LEFT JOIN jenis_layanan l ON rm.id_layanan = l.id_layanan
          LEFT JOIN data_pegawai pg ON rm.id_pegawai = pg.id_pegawai
          ORDER BY rm.tanggal_periksa DESC, rm.id_rekam DESC";

$result = $conn->query($query);

// Proses hapus rekam medis
if (isset($_GET['hapus'])) {
    $id_rekam = intval($_GET['hapus']);
    
    // Ambil data sebelum dihapus untuk backup
    $backup_query = "SELECT * FROM rekam_medis WHERE id_rekam = ?";
    $backup_stmt = $conn->prepare($backup_query);
    $backup_stmt->bind_param("i", $id_rekam);
    $backup_stmt->execute();
    $backup_data = $backup_stmt->get_result()->fetch_assoc();
    $backup_stmt->close();
    
    if ($backup_data) {
        // Insert ke backup table
        $insert_backup = "INSERT INTO backup_rekam_medis 
                         (id_rekam, id_pasien, id_layanan, keluhan, tindakan, hasil, tanggal_periksa, id_pegawai, deleted_by)
                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt_backup = $conn->prepare($insert_backup);
        $stmt_backup->bind_param("iiissssii",
            $backup_data['id_rekam'],
            $backup_data['id_pasien'],
            $backup_data['id_layanan'],
            $backup_data['keluhan'],
            $backup_data['tindakan'],
            $backup_data['hasil'],
            $backup_data['tanggal_periksa'],
            $backup_data['id_pegawai'],
            $_SESSION['id_admin']
        );
        
        if ($stmt_backup->execute()) {
            // Hapus dari tabel utama
            $delete_query = "DELETE FROM rekam_medis WHERE id_rekam = ?";
            $stmt_delete = $conn->prepare($delete_query);
            $stmt_delete->bind_param("i", $id_rekam);
            $stmt_delete->execute();
            $stmt_delete->close();
            
            $_SESSION['success'] = "Rekam medis berhasil dihapus dan disimpan di backup.";
        } else {
            $_SESSION['error'] = "Gagal membuat backup data.";
        }
        $stmt_backup->close();
    } else {
        $_SESSION['error'] = "Data tidak ditemukan.";
    }
    
    header("Location: rekam_medis.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rekam Medis</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f5f5f5;
        }

        .page-rekam-medis {
            padding: 20px;
            max-width: 1200px;
            margin: 0 auto;
        }

        h2 {
            color: #2c3e50;
            margin-bottom: 10px;
            padding-bottom: 10px;
            border-bottom: 2px solid #3498db;
        }

        p {
            color: #7f8c8d;
            margin-bottom: 20px;
        }

        /* Notifikasi */
        .alert {
            padding: 12px 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            font-weight: 500;
        }

        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        /* Tabel */
        .table-container {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            margin-top: 20px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        thead {
            background-color: #3498db;
            color: white;
        }

        th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
        }

        td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
        }

        tbody tr:hover {
            background-color: #f9f9f9;
        }

        /* Tombol Aksi */
        .btn-hapus {
            background-color: #e74c3c;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            transition: background-color 0.3s;
        }

        .btn-hapus:hover {
            background-color: #c0392b;
        }

        .btn-detail {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            text-decoration: none;
            display: inline-block;
            margin-right: 5px;
        }

        .btn-detail:hover {
            background-color: #2980b9;
        }

        /* Badge Status */
        .status-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }

        /* Search Bar */
        .search-container {
            margin: 20px 0;
            display: flex;
            gap: 10px;
        }

        .search-input {
            flex: 1;
            padding: 10px 15px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }

        .btn-search {
            background-color: #2ecc71;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 600;
        }

        .btn-search:hover {
            background-color: #27ae60;
        }

        /* Action Bar */
        .action-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding: 15px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }

        .btn-refresh {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 600;
        }

        .btn-refresh:hover {
            background-color: #2980b9;
        }

        /* Modal Detail */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }

        .modal.show {
            display: flex;
        }

        .modal-content {
            background: white;
            border-radius: 10px;
            width: 90%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
        }

        .modal-header {
            padding: 20px;
            background: #3498db;
            color: white;
            border-radius: 10px 10px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-header h3 {
            margin: 0;
        }

        .modal-close {
            background: none;
            border: none;
            color: white;
            font-size: 24px;
            cursor: pointer;
        }

        .modal-body {
            padding: 20px;
        }

        .detail-row {
            margin-bottom: 15px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }

        .detail-label {
            font-weight: 600;
            color: #555;
            margin-bottom: 5px;
        }

        .detail-value {
            color: #333;
            line-height: 1.5;
        }

        /* Responsive */
        @media (max-width:768px){.table-container {
            overflow-x: auto;
        }

        table {
            min-width: 800px;
        }

        .action-bar {
            flex-direction: column;
            gap: 10px;
        }

        .search-container {
            width: 100%;
        }
        }

        /* No data */
        .no-data {
            text-align: center;
            padding: 40px;
            color: #7f8c8d;
            font-style: italic;
        }
    </style>
</head>

<body>
    <div class="page-rekam-medis">
        <h2>🩺 Rekam Medis Pasien</h2>
        <p>Kelola dan lihat riwayat rekam medis pasien. Data yang dihapus akan disimpan di backup.</p>

        <!-- Notifikasi -->
        <?php if(isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if(isset($_SESSION['error'])): ?>
        <div class="alert alert-error">
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <!-- Action Bar -->
        <div class="action-bar">
            <div class="search-container">
                <input type="text" id="searchInput" placeholder="🔍 Cari nama pasien, keluhan, atau tindakan..."
                    class="search-input">
                <button class="btn-search" onclick="searchRecords()">Cari</button>
                <button class="btn-refresh" onclick="location.reload()">🔄 Refresh</button>
            </div>
        </div>

        <!-- Tabel Rekam Medis -->
        <div class="table-container">
            <table id="tableRekam">
                <thead>
                    <tr>
                        <th>No</th>
                        <th>Nama Pasien</th>
                        <th>Layanan</th>
                        <th>Keluhan</th>
                        <th>Tindakan</th>
                        <th>Tanggal Periksa</th>
                        <th>Dokter</th>
                        <th>Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if($result->num_rows > 0): ?>
                    <?php $no = 1; while($row = $result->fetch_assoc()): ?>
                    <tr data-id="<?php echo $row['id_rekam']; ?>">
                        <td><?php echo $no++; ?></td>
                        <td><?php echo htmlspecialchars($row['nama_pasien']); ?></td>
                        <td><?php echo htmlspecialchars($row['nama_layanan']); ?></td>
                        <td class="truncate">
                            <?php echo htmlspecialchars(substr($row['keluhan'], 0, 50)) . (strlen($row['keluhan']) > 50 ? '...' : ''); ?>
                        </td>
                        <td class="truncate">
                            <?php echo htmlspecialchars(substr($row['tindakan'], 0, 50)) . (strlen($row['tindakan']) > 50 ? '...' : ''); ?>
                        </td>
                        <td><?php echo date('d M Y', strtotime($row['tanggal_periksa'])); ?></td>
                        <td><?php echo htmlspecialchars($row['nama_pegawai']); ?></td>
                        <td>
                            <button class="btn-detail" onclick="showDetail(<?php echo $row['id_rekam']; ?>)">📄
                                Detail</button>
                            <button class="btn-hapus"
                                onclick="hapusRekam(<?php echo $row['id_rekam']; ?>, '<?php echo htmlspecialchars($row['nama_pasien']); ?>')">🗑️
                                Hapus</button>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                    <?php else: ?>
                    <tr>
                        <td colspan="8" class="no-data">Belum ada data rekam medis.</td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Modal Detail -->
    <div class="modal" id="detailModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>📄 Detail Rekam Medis</h3>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body" id="modalBody">
                <!-- Detail akan dimuat via AJAX -->
            </div>
        </div>
    </div>

    <script>
        // Fungsi untuk mencari data
        function searchRecords() {
            const searchTerm = document.getElementById('searchInput').value.toLowerCase();
            const rows = document.querySelectorAll('#tableRekam tbody tr');

            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        }

        // Fungsi untuk menampilkan detail rekam medis
        function showDetail(id_rekam) {
            // Tampilkan modal dengan loading
            const modal = document.getElementById('detailModal');
            const modalBody = document.getElementById('modalBody');
            modalBody.innerHTML = '<div style="text-align: center; padding: 20px;">Memuat data...</div>';
            modal.classList.add('show');

            // Load data via AJAX
            fetch(`admin/get_detail_rekam.php?id_rekam=${id_rekam}`)
                .then(response => response.text())
                .then(data => {
                    modalBody.innerHTML = data;
                })
                .catch(error => {
                    modalBody.innerHTML =
                        '<div style="color: red; text-align: center;">Terjadi kesalahan saat memuat data.</div>';
                });
        }

        // Fungsi untuk menutup modal
        function closeModal() {
            document.getElementById('detailModal').classList.remove('show');
        }

        // Fungsi untuk menghapus rekam medis
        function hapusRekam(id_rekam, nama_pasien) {
            if (confirm(
                    `Apakah Anda yakin ingin menghapus rekam medis untuk pasien:\n"${nama_pasien}"?\n\nData akan disimpan di backup.`
                    )) {
                window.location.href = `rekam_medis.php?hapus=${id_rekam}`;
            }
        }

        // Fungsi untuk menutup modal saat klik di luar
        window.onclick = function (event) {
            const modal = document.getElementById('detailModal');
            if (event.target == modal) {
                closeModal();
            }
        }

        // Tambahkan event listener untuk search input
        document.getElementById('searchInput').addEventListener('keyup', function (e) {
            if (e.key === 'Enter') {
                searchRecords();
            }
        });
    </script>
</body>

</html>